/**
 * \file sdc_keystore_keys.c
 *
 * \brief Functions etc. required by applications to manage
 * keystorage keys
 *
 * \author Christoph Gellner (cgellner@de.adit-jv.com)
 *
 * \copyright (c) 2015 Advanced Driver Information Technology.
 * This code is developed by Advanced Driver Information Technology.
 * Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
 * All rights reserved.
 *
 *
 ***********************************************************************/

#include <stdlib.h>
#include <unistd.h>
#include <sys/types.h>
#include <sdc_keystore_keys.h>
#include <private/sdc_arch.h>

/* Definitions types and defaults */

/* Functions */

static sdc_error_t common_insert_import_storage_key_checks(
    const sdc_keystore_key_cfg_t *key_config)
{
    sdc_error_t err = SDC_OK;
    sdc_key_len_bmsk_t keylen_bmsk;

    if (key_config->kid == NULL) {
        return SDC_KEY_INVALID;
    }

    if (key_config->common.perms == NULL) {
        return SDC_PERM_INVALID;
    }

    if ((key_config->kid_opt < SDC_CREATE_KEY_OPT_FIRST) || (key_config->kid_opt > SDC_CREATE_KEY_OPT_LAST)) {
        return SDC_INVALID_PARAMETER;
    }

    if ((key_config->key_stor_opt < SDC_STORAGE_KEY_OPT_FIRST) || (key_config->key_stor_opt > SDC_STORAGE_KEY_OPT_LAST)) {
        return SDC_INVALID_PARAMETER;
    }

    if ((key_config->kid_opt == SDC_CREATE_KEY_FIXED_ID) && (*(key_config->kid) == SDC_FLAG_INVALID_KID))
        return SDC_KEY_INVALID;

    if ((key_config->common.len < SDC_KEY_LEN_FIRST) || (key_config->common.len >= SDC_KEY_LEN_END))
        return SDC_KEYLEN_INVALID;

    if ((key_config->common.fmt < SDC_KEY_FMT_FIRST) || (key_config->common.fmt >= SDC_KEY_FMT_END))
        return SDC_KEY_FMT_INVALID;

    err = sdc_key_len_bmsks(key_config->common.fmt, &keylen_bmsk);
    if (err == SDC_OK) {
        if ((keylen_bmsk & SDC_KEY_LEN_TO_BMSK(key_config->common.len)) == 0)
            err = SDC_KEYLEN_INVALID;
    }

    return err;
}

sdc_error_t sdc_insert_storage_key(
    const sdc_key_fmt_t fmt,
    const sdc_key_len_t len,
    sdc_key_id_t *kid,
    const sdc_key_id_options_t kid_opt,
    const sdc_key_storage_options_t key_stor_opt,
    const sdc_permissions_t *permissions,
    sdc_keysecret_enc_t key_secret_enc,
    const uint8_t *key_secret,
    const size_t key_secret_len)
{
    sdc_error_t err;
    sdc_keystore_key_cfg_t key_config = {
            .kid = kid,
            .kid_opt = kid_opt,
            .key_stor_opt = key_stor_opt,
            .common = {
                    .fmt = fmt,
                    .len = len,
                    .perms = permissions,
            },
    };
    sdc_keysecret_t key_secret_st = {
            .enc = key_secret_enc,
            .secret = key_secret,
            .secret_len = key_secret_len,
    };

    err = common_insert_import_storage_key_checks(
        &key_config);

    if ((err == SDC_OK) && ((key_secret == NULL) || (key_secret_len == 0)))
        err = SDC_KEY_INVALID;

    if (err == SDC_OK)
        err = sdc_intern_check_permission_range(permissions);

    if (err == SDC_OK)
        err = sdc_intern_check_key_enc(key_secret_enc, fmt);

    if (err == SDC_OK)
        err = sdc_arch_insert_storage_key(&key_config, &key_secret_st);

    return err;
}

sdc_error_t sdc_insert_plain_storage_key(
    sdc_key_id_t *kid,
    const sdc_key_id_options_t kid_opt,
    const uint8_t *plain_key,
    const size_t plain_key_len,
    const sdc_key_storage_options_t key_stor_opt,
    const sdc_permissions_t *permissions)
{
    sdc_error_t err;
    sdc_key_len_t keylen;

    err = sdc_key_len_from_bytes(plain_key_len, &keylen);

    if (err == SDC_OK) {
        err = sdc_insert_storage_key(
            SDC_KEY_FMT_SIMPLE_SYM, keylen,
            kid, kid_opt,
            key_stor_opt,
            permissions,
            SDC_KEY_ENC_PLAIN,
            plain_key, plain_key_len);
    }

    return err;
}

static sdc_error_t sdc_generate_sym_storage_key(
    const sdc_key_fmt_t fmt,
    const sdc_key_len_t len,
    sdc_key_id_t *kid,
    const sdc_key_id_options_t kid_opt,
    const sdc_key_storage_options_t key_stor_opt,
    const sdc_permissions_t *permissions)
{
    sdc_error_t err;
    sdc_keystore_key_cfg_t key_config;

    key_config.kid = kid;
    key_config.kid_opt = kid_opt;
    key_config.key_stor_opt = key_stor_opt;
    key_config.common.fmt = fmt;
    key_config.common.len = len;
    key_config.common.perms = permissions;

    err = common_insert_import_storage_key_checks(
        &key_config);

    if (err == SDC_OK)
        err = sdc_intern_check_permission_range(permissions);

    if (err == SDC_OK)
        err = sdc_intern_check_sym_key_fmt(fmt);

    if (err == SDC_OK)
        err = sdc_arch_generate_sym_storage_key(
            &key_config);

    return err;
}

sdc_error_t sdc_generate_random_storage_key(
    sdc_key_id_t *kid,
    const sdc_key_id_options_t kid_opt,
    const size_t bytes,
    const sdc_key_storage_options_t key_stor_opt,
    const sdc_permissions_t *permissions)
{
    sdc_error_t err;
    sdc_key_len_t keylen;

    err = sdc_key_len_from_bytes(bytes, &keylen);

    if (err == SDC_OK) {
        err = sdc_generate_sym_storage_key(
            SDC_KEY_FMT_SIMPLE_SYM, keylen,
            kid, kid_opt,
            key_stor_opt,
            permissions);
    }

    return err;
}

sdc_error_t sdc_import_wrapped_storage_key(sdc_session_t *session,
                                           const sdc_key_fmt_t fmt,
                                           const sdc_key_len_t len,
                                           sdc_key_id_t *kid,
                                           const sdc_key_id_options_t kid_opt,
                                           const sdc_key_storage_options_t key_stor_opt,
                                           const sdc_permissions_t *permissions,
                                           const sdc_wrap_unwrap_type_t *wrap_type,
                                           sdc_keysecret_enc_t wrapped_key_enc,
                                           const uint8_t *wrap_iv, const size_t wrap_iv_len,
                                           const uint8_t *wrap_tag, const size_t wrap_tag_len,
                                           const uint8_t *wrapped_key, const size_t wrapped_key_len)
{
    sdc_error_t err = SDC_OK;
    sdc_wrap_unwrap_desc_t internal_desc;
    sdc_keystore_key_cfg_t key_config = {
            .kid = kid,
            .kid_opt = kid_opt,
            .key_stor_opt = key_stor_opt,
            .common = {
                    .fmt = fmt,
                    .len = len,
                    .perms = permissions,
            },
    };
    sdc_wrapped_key_t wrapped_key_st = {
            .wrap_type = (sdc_wrap_unwrap_type_t *)wrap_type,
            .wrap_desc = &internal_desc,
            .enc = wrapped_key_enc,
            .iv = (uint8_t *)wrap_iv,
            .iv_len = wrap_iv_len,
            .tag = (uint8_t *)wrap_tag,
            .tag_len = wrap_tag_len,
            .wrapped_data = (uint8_t *)wrapped_key,
            .wrapped_len = wrapped_key_len,
    };

    if (!session)
        err = SDC_SESSION_INVALID;

    if (err == SDC_OK)
        err = common_insert_import_storage_key_checks(
            &key_config);

    if (err == SDC_OK)
        err = sdc_intern_check_permission_range(permissions);

    if (err == SDC_OK)
        err = sdc_intern_check_key_enc(wrapped_key_enc, fmt);

    if ((err == SDC_OK) &&
        (SDC_OK != sdc_intern_check_tag_iv_input_buffer(wrap_tag, wrap_tag_len, SDC_TAG_USE_DEFAULT)))
        err = SDC_TAG_DATA_INVALID;

    if ((err == SDC_OK) &&
        (SDC_OK != sdc_intern_check_tag_iv_input_buffer(wrap_iv, wrap_iv_len, SDC_IV_USE_DEFAULT)))
        err = SDC_IV_INVALID;

    if ((err == SDC_OK) &&
        (SDC_OK != sdc_intern_check_data_input_buffer(wrapped_key, wrapped_key_len)))
        err = SDC_IN_DATA_INVALID;

    if (err == SDC_OK)
        err = sdc_wrap_unwrap_desc_fill(session, wrap_type, &internal_desc, wrapped_key_len);

    if (err == SDC_OK)
        err = sdc_arch_import_wrapped_storage_key(
                session, &key_config, &wrapped_key_st);

    return err;
}

sdc_error_t sdc_import_formatted_storage_key(sdc_session_t *session,
                                             const sdc_key_fmt_t fmt,
                                             const sdc_key_len_t len,
                                             const sdc_keysecret_enc_t key_enc,
                                             sdc_key_id_t *kid,
                                             const sdc_key_id_options_t kid_opt,
                                             const sdc_key_storage_options_t key_stor_opt,
                                             const sdc_permissions_t *permissions,
                                             const uint8_t *formatted_key, const size_t formatted_key_len)
{
    sdc_error_t err = SDC_OK;
    sdc_error_t tmperr;
    sdc_form_header_generic_t form_header;
    sdc_form_header_wrap_unwrap_generic_t *form_wrapped_key = NULL;
    sdc_wrap_unwrap_type_t *wrap_type = NULL;

    if (SDC_OK != sdc_intern_check_data_input_buffer(formatted_key, formatted_key_len))
        err = SDC_FORMATTED_DATA_INVALID;

    if (!session)
        err = SDC_SESSION_INVALID;

    if (err != SDC_OK)
        return err;

    err = sdc_intern_formatted_read_header (
        &form_header,
        formatted_key, formatted_key_len);

    if (err == SDC_OK) {
        switch (form_header.operation) {
        case SDC_FORMATTED_TYPE_WRAP_UNWRAP:
            /* legacy wrapped key */
            form_wrapped_key = &form_header.wrap_unwrap;
            /* we need to typecast as the same pointers in wrap struct are used for
             * read and write. Before calling unwrap we will make it const again */
            err = sdc_intern_wrap_unwrap_formatted_update_pointers(
                &form_header,
                (uint8_t*)formatted_key);

            /* get alg and block mode */
            if (err == SDC_OK) {
                err = sdc_wrap_unwrap_type_alloc(&wrap_type);
            }
            if (err == SDC_OK) {
                err = sdc_wrap_unwrap_type_set_algorithm(wrap_type, form_wrapped_key->alg);
            }
            if (err == SDC_OK) {
                err = sdc_wrap_unwrap_type_set_block_mode(wrap_type, form_wrapped_key->blk);
            }

            if (err == SDC_OK) {
                err = sdc_import_wrapped_storage_key(session,
                                                     fmt, len,
                                                     kid, kid_opt, key_stor_opt,
                                                     permissions,
                                                     wrap_type, key_enc,
                                                     form_wrapped_key->iv, form_wrapped_key->iv_len,
                                                     form_wrapped_key->tag, form_wrapped_key->tag_len,
                                                     form_wrapped_key->data, form_wrapped_key->data_len);
            }
            break;
        // TODO : Add special format for keys
        default:
            err = SDC_FORMATTED_DATA_INVALID;
        }
    }

    if (wrap_type != NULL) {
        tmperr = sdc_wrap_unwrap_type_free(wrap_type);

        if (err == SDC_OK)
            err = tmperr;
    }

    return err;
}

sdc_error_t sdc_remove_storage_key(sdc_key_id_t kid)
{
    if (kid == SDC_FLAG_INVALID_KID)
        return SDC_KEY_INVALID;

    return sdc_arch_remove_storage_key(kid);
}
